' sap-script-backlog.vbs - Tool Backlog data extraction from SAP
On Error Resume Next

' Set scripting mode to true
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting warnings in registry
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\FileWriteAccess", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\FileDirWriteAccess", 0, "REG_DWORD"

' Get parameters from command line
Dim plant, storageLocation, dataOutputPath, tmpOutputPath
plant = ""
storageLocation = ""

' Get application folder
Set fso = CreateObject("Scripting.FileSystemObject")
Dim appFolder, scriptPath, programFolder
scriptPath = WScript.ScriptFullName
appFolder = fso.GetParentFolderName(scriptPath)
programFolder = fso.GetParentFolderName(fso.GetParentFolderName(appFolder))

' Debug info - echo paths to help diagnose where files are actually being saved
WScript.Echo "DEBUG:scriptPath=" & scriptPath
WScript.Echo "DEBUG:appFolder=" & appFolder
WScript.Echo "DEBUG:programFolder=" & programFolder

' Create tmp folder if it doesn't exist
tmpOutputPath = programFolder & "\tmp"
If Not fso.FolderExists(tmpOutputPath) Then
    On Error Resume Next
    fso.CreateFolder(tmpOutputPath)
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not create tmp folder: " & Err.Description
        ' Fall back to temp folder
        tmpOutputPath = fso.GetSpecialFolder(2) ' Temp folder
        WScript.Echo "DEBUG:Falling back to: " & tmpOutputPath
    End If
    On Error GoTo 0
End If

WScript.Echo "DEBUG:Using tmpOutputPath=" & tmpOutputPath

' Check if parameters are provided
If WScript.Arguments.Count > 0 Then
    plant = WScript.Arguments.Item(0)
End If

If WScript.Arguments.Count > 1 Then
    storageLocation = WScript.Arguments.Item(1)
End If

If WScript.Arguments.Count > 2 Then
    dataOutputPath = WScript.Arguments.Item(2)
Else
    ' Default to tmp folder in program directory
    dataOutputPath = tmpOutputPath & "\sap_backlog_data.txt"
End If

' Declare log file object and path
Dim logFile, logFilePath

' Initialize log file if path is provided
If WScript.Arguments.Count > 3 Then
    logFilePath = WScript.Arguments.Item(3)
    ' Try to open the log file in append mode (8)
    On Error Resume Next
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script start
        LogToFile "========================================="
        LogToFile "Backlog Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Plant: " & plant
        LogToFile "Storage Location: " & storageLocation
        LogToFile "Data Output Path: " & dataOutputPath
        LogToFile "Temp Output Path: " & tmpOutputPath
        LogToFile "========================================="
    End If
End If

' Function to log message to file
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Function to log message to both console and file
Sub LogMessage(message)
    WScript.Echo message
    LogToFile message
End Sub

' Initialize data output file
Dim outputFile, dataCollected
Set outputFile = fso.CreateTextFile(dataOutputPath, True)
dataCollected = False

' Initialize variables for file paths
Dim ih10FilePath, zstkFilePath
ih10FilePath = tmpOutputPath & "\IH10_data.txt"
zstkFilePath = tmpOutputPath & "\ZSTK_data.txt"

On Error Resume Next

' Using the simplified SAP connection approach
If Not IsObject(application) Then
   Set SapGuiAuto = GetObject("SAPGUI")
   Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If
If Not IsObject(session) Then
   Set session = connection.Children(0)
End If
If IsObject(WScript) Then
   WScript.ConnectObject session, "on"
   WScript.ConnectObject application, "on"
End If

LogMessage "Starting backlog data collection..."

' Ensure window is maximized
session.findById("wnd[0]").maximize

' === Run IH10 (Stock Overview) ===
LogMessage "Opening IH10 transaction..."
session.findById("wnd[0]/tbar[0]/okcd").text = "/nIH10"
session.findById("wnd[0]").sendVKey 0

' Set filters for IH10
If plant <> "" Then
    LogMessage "Setting plant: " & plant
    
    ' Check if comma-separated list
    If InStr(plant, ",") > 0 Then
        LogMessage "Multiple plants detected: " & plant
        plantArray = Split(plant, ",")
        
        ' Use multi-selection button
        session.findById("wnd[0]/usr/btn%_WERK_%_APP_%-VALU_PUSH").press
        
        ' Add each plant from the list
        For i = 0 To UBound(plantArray)
            currPlant = Trim(plantArray(i))
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currPlant
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currPlant)
        Next
        
        ' Confirm selection
        session.findById("wnd[1]/tbar[0]/btn[8]").press
    Else
        ' Single plant - set directly
        session.findById("wnd[0]/usr/ctxtWERK-LOW").text = plant
        session.findById("wnd[0]/usr/ctxtWERK-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtWERK-LOW").caretPosition = Len(plant)
    End If
End If

If storageLocation <> "" Then
    LogMessage "Setting storage location: " & storageLocation
    
    ' Check if comma-separated list
    If InStr(storageLocation, ",") > 0 Then
        LogMessage "Multiple storage locations detected: " & storageLocation
        slocArray = Split(storageLocation, ",")
        
        ' Use multi-selection button
        session.findById("wnd[0]/usr/btn%_LAGER_%_APP_%-VALU_PUSH").press
        
        ' Add each SLOC from the list
        For i = 0 To UBound(slocArray)
            currSloc = Trim(slocArray(i))
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currSloc
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currSloc)
        Next
        
        ' Confirm selection
        session.findById("wnd[1]/tbar[0]/btn[8]").press
    Else
        ' Single SLOC - set directly
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").text = storageLocation
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").caretPosition = Len(storageLocation)
    End If
End If

' Execute IH10 with variant
LogMessage "Executing IH10 with variant"
session.findById("wnd[0]/usr/ctxtVARIANT").text = "/BCKLGFRTVDR"
session.findById("wnd[0]/usr/ctxtVARIANT").setFocus
session.findById("wnd[0]/usr/ctxtVARIANT").caretPosition = 12
session.findById("wnd[0]/tbar[1]/btn[8]").press

' Export IH10 data
LogMessage "Exporting IH10 data..."
On Error Resume Next
session.findById("wnd[0]/mbar/menu[0]/menu[10]/menu[2]").select

' If first menu path fails, try alternate path
If Err.Number <> 0 Then
    Err.Clear
    session.findById("wnd[0]/mbar/menu[0]/menu[3]/menu[1]").select
End If

' Set export options
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").setFocus
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Set export path and filename
session.findById("wnd[1]/usr/ctxtDY_PATH").text = tmpOutputPath
session.findById("wnd[1]/usr/ctxtDY_FILENAME").text = "IH10_data.txt"
session.findById("wnd[1]/usr/ctxtDY_FILENAME").caretPosition = 13
session.findById("wnd[1]/tbar[0]/btn[11]").press

' Return to home screen
WScript.Sleep 2000
session.findById("wnd[0]").sendVKey 3
session.findById("wnd[0]").sendVKey 3

' === Run ZSTK (Custom Report) ===
LogMessage "Opening ZSTK transaction..."
session.findById("wnd[0]/tbar[0]/okcd").text = "/nZSTK"
session.findById("wnd[0]").sendVKey 0

' Set filters for ZSTK
session.findById("wnd[0]/usr/ctxtSO_MATNR-LOW").text = "" ' Clear material number

Dim isMultiplePlantsZSTK, isMultipleSlocsZSTK
Dim plantArrayZSTK, slocArrayZSTK ' Use ZSTK-specific array names

isMultiplePlantsZSTK = False
isMultipleSlocsZSTK = False

' Step 1: Populate the single plant field if plant is provided
If plant <> "" Then
    LogMessage "Processing plant for ZSTK: " & plant
    If InStr(plant, ",") > 0 Then
        isMultiplePlantsZSTK = True
        plantArrayZSTK = Split(plant, ",")
        If UBound(plantArrayZSTK) >= 0 Then
            Dim firstPlantZSTK
            firstPlantZSTK = Trim(plantArrayZSTK(0))
            session.findById("wnd[0]/usr/ctxtSO_WERKS-LOW").text = firstPlantZSTK
            LogMessage "Set first plant '" & firstPlantZSTK & "' in ZSTK main field."
        Else
            LogMessage "Warning: Plant string '" & plant & "' resulted in an empty array for ZSTK."
            isMultiplePlantsZSTK = False ' Treat as no valid multiple plants
        End If
    Else
        ' Single plant
        session.findById("wnd[0]/usr/ctxtSO_WERKS-LOW").text = plant
        LogMessage "Set single plant '" & plant & "' in ZSTK main field."
    End If
Else
    LogMessage "No plant provided for ZSTK. ZSTK might require a plant."
    ' If ZSTK strictly requires a plant and none is given, it might error here or when executing.
    ' Current logic proceeds, assuming a variant might handle it or it's optional.
End If

' Step 2: Populate the single storage location field if storageLocation is provided
If storageLocation <> "" Then
    LogMessage "Processing storage location for ZSTK: " & storageLocation
    If InStr(storageLocation, ",") > 0 Then
        isMultipleSlocsZSTK = True
        slocArrayZSTK = Split(storageLocation, ",")
        If UBound(slocArrayZSTK) >= 0 Then
            Dim firstSlocZSTK
            firstSlocZSTK = Trim(slocArrayZSTK(0))
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").text = firstSlocZSTK
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").setFocus ' Important for some SAP screens
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").caretPosition = Len(firstSlocZSTK)
            LogMessage "Set first SLOC '" & firstSlocZSTK & "' in ZSTK main field."
        Else
            LogMessage "Warning: Storage location string '" & storageLocation & "' resulted in an empty array for ZSTK."
            isMultipleSlocsZSTK = False ' Treat as no valid multiple SLOCs
        End If
    Else
        ' Single SLOC
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").text = storageLocation
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").caretPosition = Len(storageLocation)
        LogMessage "Set single SLOC '" & storageLocation & "' in ZSTK main field."
    End If
Else
    LogMessage "No storage location provided for ZSTK. ZSTK might require a storage location."
    ' If ZSTK strictly requires a SLOC and none is given, it might error here or when executing.
End If

' Step 3: Handle multi-selection for remaining plants (if any)
' This is done AFTER both single fields have been potentially populated.
If isMultiplePlantsZSTK And UBound(plantArrayZSTK) > 0 Then ' More than one plant in the original list
    LogMessage "Opening multi-selection for remaining " & UBound(plantArrayZSTK) & " plants in ZSTK."
    session.findById("wnd[0]/usr/btn%_SO_WERKS_%_APP_%-VALU_PUSH").press
    
    Dim currentPlantZSTKLoop
    ' The first plant (plantArrayZSTK(0)) is already in the main field and usually pre-fills the first line of multi-select.
    ' We add plantArrayZSTK(1) to table row 1, plantArrayZSTK(2) to table row 2, etc.
    ' The table in the dialog is 0-indexed for rows in scripting, but the first entry from the main field populates row 0.
    For i = 1 To UBound(plantArrayZSTK) ' Start from the second item (index 1 of original array)
        currentPlantZSTKLoop = Trim(plantArrayZSTK(i))
        If currentPlantZSTKLoop <> "" Then
            ' The row index in the dialog table will be 'i' because row 0 is taken by the first plant.
            LogMessage "Adding plant '" & currentPlantZSTKLoop & "' to ZSTK multi-select dialog at table row " & i
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currentPlantZSTKLoop
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currentPlantZSTKLoop)
        End If
    Next
    
    LogMessage "Confirming ZSTK plant multi-selection."
    session.findById("wnd[1]/tbar[0]/btn[8]").press ' Confirm selection
ElseIf isMultiplePlantsZSTK And UBound(plantArrayZSTK) = 0 Then
    LogMessage "Only one plant detected for ZSTK (from '" & plant & "'), already set in main field. No multi-selection needed for plant."
End If


' Step 4: Handle multi-selection for remaining SLOCs (if any)
' This is done AFTER both single fields have been potentially populated.
If isMultipleSlocsZSTK And UBound(slocArrayZSTK) > 0 Then ' More than one SLOC in the original list
    LogMessage "Opening multi-selection for remaining " & UBound(slocArrayZSTK) & " SLOCs in ZSTK."
    session.findById("wnd[0]/usr/btn%_SO_LGORT_%_APP_%-VALU_PUSH").press
    
    Dim currentSlocZSTKLoop
    ' Similar to plants, the first SLOC is in the main field.
    ' slocArrayZSTK(1) goes to dialog table row 1, etc.
    For i = 1 To UBound(slocArrayZSTK) ' Start from the second item (index 1 of original array)
        currentSlocZSTKLoop = Trim(slocArrayZSTK(i))
        If currentSlocZSTKLoop <> "" Then
            LogMessage "Adding SLOC '" & currentSlocZSTKLoop & "' to ZSTK multi-select dialog at table row " & i
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currentSlocZSTKLoop
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currentSlocZSTKLoop)
        End If
    Next
    
    ' Confirm selection - Add proper error handling
    LogMessage "Confirming ZSTK SLOC multi-selection."
    On Error Resume Next
    session.findById("wnd[1]/tbar[0]/btn[8]").press
    If Err.Number <> 0 Then
        LogMessage "Error confirming ZSTK SLOC selection with btn[8]: " & Err.Description
        Err.Clear
        LogMessage "Trying alternative confirm button btn[0] for ZSTK SLOC."
        session.findById("wnd[1]/tbar[0]/btn[0]").press
        If Err.Number <> 0 Then
            LogMessage "Failed alternative confirm button btn[0] for ZSTK SLOC: " & Err.Description
        End If
    End If
    On Error GoTo 0
ElseIf isMultipleSlocsZSTK And UBound(slocArrayZSTK) = 0 Then
    LogMessage "Only one SLOC detected for ZSTK (from '" & storageLocation & "'), already set in main field. No multi-selection needed for SLOC."
End If

' Execute ZSTK report
LogMessage "Executing ZSTK search..."
session.findById("wnd[0]/tbar[1]/btn[8]").press

' Get layout for export
session.findById("wnd[0]/tbar[1]/btn[32]").press

' Select specific columns for export
On Error Resume Next
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").currentCellRow = 2
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").selectedRows = "2"
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").currentCellRow = 7
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").selectedRows = "7"
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Export ZSTK data
LogMessage "Exporting ZSTK data..."
session.findById("wnd[0]/mbar/menu[0]/menu[1]/menu[2]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").setFocus
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Set export path and filename
session.findById("wnd[1]/usr/ctxtDY_PATH").text = tmpOutputPath
session.findById("wnd[1]/usr/ctxtDY_FILENAME").text = "ZSTK_data.txt"
session.findById("wnd[1]/usr/ctxtDY_FILENAME").caretPosition = 13
session.findById("wnd[1]/tbar[0]/btn[11]").press

' Wait for export to complete and return to home screen
WScript.Sleep 2000
session.findById("wnd[0]").sendVKey 3
session.findById("wnd[0]").sendVKey 3

' Process the exported data files
LogMessage "Processing exported data..."
On Error Resume Next

If fso.FileExists(ih10FilePath) And fso.FileExists(zstkFilePath) Then
    LogMessage "Both export files found. Processing data..."
    
    ' Read the exported files
    Dim ih10File, zstkFile
    Dim ih10Line, zstkLine, ih10Fields, zstkFields, i, j
    
    ' Create dictionaries to store the records
    Dim ih10Records, zstkRecords, zstkData
    Set ih10Records = CreateObject("Scripting.Dictionary")
    Set zstkRecords = CreateObject("Scripting.Dictionary")
    
    ' Open files
    Set ih10File = fso.OpenTextFile(ih10FilePath, 1)
    Set zstkFile = fso.OpenTextFile(zstkFilePath, 1)
    
    ' Skip headers (skip two lines - including blank line)
    ih10File.SkipLine
    ih10File.SkipLine
    zstkFile.SkipLine
    zstkFile.SkipLine
    
    LogMessage "Reading IH10 data..."
    ' Read IH10 data
    Dim ih10Count
    ih10Count = 0
    
    Do Until ih10File.AtEndOfStream
        ih10Line = ih10File.ReadLine
        
        If Trim(ih10Line) <> "" Then
            ih10Fields = Split(ih10Line, vbTab)
            
            ' Debug information for actual field values
            If ih10Count < 3 Then ' Just log a few rows for debugging
                LogMessage "IH10 Row " & ih10Count & " field count: " & UBound(ih10Fields) + 1
                For debugIdx = 0 To UBound(ih10Fields)
                    LogMessage "  Field[" & debugIdx & "]: " & ih10Fields(debugIdx)
                Next
            End If
            
            ' Column indexes in the actual file (may need adjustment):
            ' [0]=empty, [1]=Material, [2]=SerialNumber, [3]=Equipment, [4]=ChangedOn, [5]=Plant, [6]=SLoc
            If UBound(ih10Fields) >= 6 Then
                Dim material, serialNum, equipment, plantVal, slocVal, changedDate, key
                
                material = Trim(ih10Fields(1))
                serialNum = Trim(ih10Fields(2))
                equipment = Trim(ih10Fields(3))
                plantVal = Trim(ih10Fields(5))
                slocVal = Trim(ih10Fields(6))
                
                ' Create a composite key to uniquely identify each record
                key = material & "|" & plantVal & "|" & slocVal & "|" & serialNum
                
                ' Create a new Dictionary for this record
                Dim ih10Record
                Set ih10Record = CreateObject("Scripting.Dictionary")
                
                ih10Record.Add "Material", material
                ih10Record.Add "SerialNumber", serialNum
                ih10Record.Add "Equipment", equipment
                ih10Record.Add "Plant", plantVal
                ih10Record.Add "SLOC", slocVal
                ih10Record.Add "BinLocation", ""
                
                ' Convert date format from DD.MM.YYYY to YYYYMMDD
                Dim dateStr, dateParts
                dateStr = Trim(ih10Fields(4)) ' ChangedOn column
                
                If dateStr <> "" And InStr(dateStr, ".") > 0 Then
                    dateParts = Split(dateStr, ".")
                    If UBound(dateParts) = 2 Then
                        dateStr = dateParts(2) & dateParts(1) & dateParts(0)
                    End If
                End If
                
                ih10Record.Add "ChangedDate", dateStr
                
                ' Add to the collection
                If Not ih10Records.Exists(key) Then
                    ih10Records.Add key, ih10Record
                    ih10Count = ih10Count + 1
                End If
            End If
        End If
    Loop
    
    LogMessage "Read " & ih10Count & " records from IH10"
    
    LogMessage "Reading ZSTK data..."
    ' Read ZSTK data
    Dim zstkCount
    zstkCount = 0
    
    Do Until zstkFile.AtEndOfStream
        zstkLine = zstkFile.ReadLine
        
        If Trim(zstkLine) <> "" Then
            zstkFields = Split(zstkLine, vbTab)
            
            ' Debug information for actual field values
            If zstkCount < 3 Then ' Just log a few rows for debugging
                LogMessage "ZSTK Row " & zstkCount & " field count: " & UBound(zstkFields) + 1
                For debugIdx = 0 To UBound(zstkFields)
                    LogMessage "  Field[" & debugIdx & "]: " & zstkFields(debugIdx)
                Next
            End If
            
            ' Column indexes in the actual file:
            ' [0]=empty, [1]=Material, [2]=MaterialDesc, [3]=Plant, [4]=SLoc, [5]=Bin, [6]=Profl., [7]=Unrestr.
            If UBound(zstkFields) >= 6 Then
                Dim zstkMaterial, zstkPlant, zstkSloc, zstkBin, zstkProfl, zstkKey
                
                zstkMaterial = Trim(zstkFields(1))
                zstkPlant = Trim(zstkFields(3))
                zstkSloc = Trim(zstkFields(4))
                zstkBin = Trim(zstkFields(5))
                zstkProfl = ""
                If UBound(zstkFields) >= 6 Then
                    zstkProfl = Trim(zstkFields(6))
                End If
                
                ' Create a composite key that can be matched with IH10 data
                zstkKey = zstkMaterial & "|" & zstkPlant & "|" & zstkSloc
                
                ' Store bin location and profl for matching with IH10 data
                If Not zstkRecords.Exists(zstkKey) Then
                    ' Store as a dictionary with bin and profl
                    Set zstkData = CreateObject("Scripting.Dictionary")
                    zstkData.Add "Bin", zstkBin
                    zstkData.Add "Profl", zstkProfl
                    zstkRecords.Add zstkKey, zstkData
                    zstkCount = zstkCount + 1
                End If
            End If
        End If
    Loop
    
    LogMessage "Read " & zstkCount & " records from ZSTK"
    
    ' Close files
    ih10File.Close
    zstkFile.Close
    
    ' Write header to output file
    outputFile.WriteLine "Material|Serial Number|Equipment Number|Plant|SLOC|Profl.|Bin Location|Changed Date"
    
    LogMessage "Merging data and writing output..."
    ' Process and merge data
    Dim outputCount
    outputCount = 0
    
    ' Loop through IH10 records and find matching bin locations and profl from ZSTK
    For Each key In ih10Records.Keys
        Dim rec, materialNum, serialNo, equipNum, plantCode, slocCode, binLoc, proflVal, changDate
        Dim parts, materialKey
        
        Set rec = ih10Records(key)
        
        materialNum = rec("Material")
        serialNo = rec("SerialNumber")
        equipNum = rec("Equipment")
        plantCode = rec("Plant")
        slocCode = rec("SLOC")
        changDate = rec("ChangedDate")
        
        ' Create a key to look up bin location and profl in ZSTK records
        materialKey = materialNum & "|" & plantCode & "|" & slocCode
        
        ' Look for matching data in ZSTK
        binLoc = ""
        proflVal = ""
        If zstkRecords.Exists(materialKey) Then
            Set zstkData = zstkRecords(materialKey)
            binLoc = zstkData("Bin")
            proflVal = zstkData("Profl")
        End If
        
        ' Write to output file
        outputFile.WriteLine materialNum & "|" & _
                             serialNo & "|" & _
                             equipNum & "|" & _
                             plantCode & "|" & _
                             slocCode & "|" & _
                             proflVal & "|" & _
                             binLoc & "|" & _
                             changDate
        
        outputCount = outputCount + 1
    Next
    
    LogMessage "Wrote " & outputCount & " records to output file"
    dataCollected = (outputCount > 0)
Else
    LogMessage "One or both export files are missing"
    If Not fso.FileExists(ih10FilePath) Then
        LogMessage "Missing: " & ih10FilePath
    End If
    If Not fso.FileExists(zstkFilePath) Then
        LogMessage "Missing: " & zstkFilePath
    End If
End If

' Close output file
outputFile.Close

' Return results
If dataCollected Then
    LogMessage "SUCCESS: Data collection completed successfully"
    WScript.Echo "SUCCESS:" & dataOutputPath
Else
    LogMessage "WARNING: No data was collected"
    WScript.Echo "WARNING: No data was collected"
End If

' Close log file at end of script
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

CloseLogFile()